/**
 * @fileOverview シスケット共通ライブラリ。
 * @name SysketLibrary.js
 * @author 稲垣　順治
 * @version 1.0.0
 */

/**
 * @namespace
 * 全てstaticなメソッドばかりを持つ4つのクラスを持つ。
 * @example
 * いずれも<br />
 * 　　flg = SysketLibrary.CommonUtil.isEmpty(obj);<br />
 * のように使用することができます。記述が長すぎて面倒な場合は<br />
 * 　　var cUtil = SysketLibrary.CommonUtil;
 * 　　flg = cUtil.isEmpty(obj);<br />
 * のように記述するとよいでしょう。
 */
var SysketLibrary = {
    version:'1.0.0'
};

/**
 * @lends SysketLibrary
 */
(function (){

	/**
	 * @class
	 * 共通ユーティリティメソッドを提供する。<br />
	 * （メソッドはいずれもstaticで、本クラスをnewする必要はありません）
	 */
	this.CommonUtil = new function() {

		/**
		 * objが空(undefined, null 空文字)かどうかの判定
		 * @function
		 * @param obj 対象オブジェクト
		 * @returns true:空、false:空ではない
		 */
		this.isEmpty = function(obj) {
			if(obj == null) {
				return true;
			}

			if(obj == undefined) {
				return true;
			}

			if(obj === "") {
				return true;
			}

			return false;
		};

		/**
		 * objが空(undefined, null 空文字)かどうかの判定
		 * @function
		 * @param obj 対象オブジェクト
		 * @returns true:空、false:空ではない
		 */
		this.emptyTo = function(obj1, obj2) {
			if(this.isEmpty(obj1)) {
				return obj2;
			}

			return obj1;
		};


		/**
		 * objが配列かどうかの判定
		 * @function
		 * @param obj 対象オブジェクト
		 * @returns true:配列、false:配列ではない
		 */
		this.isArray = function(obj) {
			if(this.isEmpty(obj)) {
				return false;
			}

			return obj instanceof Array;
		};

		/**
		 * アドレスから"?"以降の文字列を配列で返す
		 * @function
		 * @returns 整形後の配列
		 */
		this.getGetArray = function() {
			var getArray = new Array();
			var query = window.location.search.substring(1);
			var parms = query.split('&');

			for (var i=0; i<parms.length; i++) {
				var pos = parms[i].indexOf('=');
				if (pos > 0) {
					var key = parms[i].substring(0,pos);
					var val = parms[i].substring(pos+1);
					getArray[key] = val;
				}
			}

			return getArray instanceof Array;
		};


		/**
		 * strが配列aryの要素で始まっていれば最初のその要素を返す
		 * aryは配列でなくとも通常の文字列でもよい
		 * @function
		 * @param str 対象文字列
		 * @param ary チェックする文字列の配列、または文字列
		 * @returns ヒットした文字列、なければfalse
		 */
		this.parseQueryVariable = function() {
			var query = window.location.search.substring(1);
			var vars = query.split("&");
			var ret = [];
			var pair;

			for(var i = 0; i < vars.length; i++) {
				pair = vars[i].split("=");
				ret[pair[0]] = pair[1];
			}

			return ret;
		};
	};

	/**
	 * @class
	 * 文字列関連のユーティリティメソッドを提供する。<br />
	 * （メソッドはいずれもstaticで、本クラスをnewする必要はありません）
	 */
	this.StringUtil = new function() {

		/**
		 * 左端をchで埋めてnum桁に整形した文字列を返す
		 * @function
		 * @param str 対象文字
		 * @param ch 埋める文字
		 * @param num 埋める文字数
		 * @returns 整形後の文字列
		 */
		this.padLeft = function(str, ch, num) {
			var ret = this.repeat(ch, num) + str;
			ret = this.right(ret, num);

			return ret;
		};

		/**
		 * 右端をchで埋めてnum桁に整形した文字列を返す
		 * @function
		 * @param str 対象文字
		 * @param ch 埋める文字
		 * @param num 埋める文字数
		 * @returns 整形後の文字列
		 */
		this.padRight = function(str, ch, num) {
			var ret = str + this.repeat(ch, num);
			ret = ret.substring(0, num);

			return ret;
		};

		/**
		 * chでnum回繰り替えした文字列を返す
		 * @function
		 * @param ch 繰り替えす文字
		 * @param num 繰り替えす回数
		 * @returns chでnum回繰り替えした文字列
		 */
		this.repeat = function(ch, num) {
			var ret = "";

			for(var i = 0; i < num; i++) {
				ret += ch;
			}

			return ret;
		};

		/**
		 * strの右端からnum文字切り出した文字列を返す
		 * @function
		 * @param str 対象文字列
		 * @param num 文字数
		 * @returns strの右端からnum文字切り出した文字列
		 */
		this.right = function(str, num) {
			if(CommonUtil.isEmpty(str)) {
				return str;
			}

			//数値等が引数の場合もあるので文字列化
			str = "" + str;

			if(str.length <= num) {
				return str;
			}

			return str.substring(str.length - num);
		};

		/**
		 * strのstartの位置から空白をスキップした最初の位置を返す
		 * @function
		 * @param str 対象文字列
		 * @param start 開始位置
		 * @returns 空白をスキップした最初の位置。startがstrの長さより大きい場合はstart。
		 */
		this.skipSpace = function(str, start) {
			if(CommonUtil.isEmpty(str)) {
				return start;
			}

			var pos = start;
			var len = str.length;

			while(pos < len) {
				if(str.substring(pos, pos + 1) != " ") {
					break;
				}

				pos++;
			}

			return pos;
		};

		/**
		 * strが配列aryの要素で始まっていれば最初のその要素を返す
		 * aryは配列でなくとも通常の文字列でもよい
		 * @function
		 * @param str 対象文字列
		 * @param ary チェックする文字列の配列、または文字列
		 * @returns ヒットした文字列、なければfalse
		 */
		this.startWith = function(str, ary) {
			if(CommonUtil.isEmpty(str)) {
				return false;
			}

			var ret = false;
			var chk = null;

			if(!CommonUtil.isArray(ary)) {
				ary = new Array(ary);
			}

			for(var i = 0; i < ary.length; i++) {
				chk = "" + ary[i];

				if(str.substring(0, chk.length) == chk) {
					ret = chk;
					break;
				}
			}

			return ret;
		};
	};


	/**
	 * @class
	 * 数値関連のユーティリティメソッドを提供する。<br />
	 * （メソッドはいずれもstaticで、本クラスをnewする必要はありません）
	 */
	this.NumberUtil = new function() {

		/**
		 * strを数値変換する。できない場合はsubを返す。
		 * @function
		 * @param str 対象文字列
		 * @param sub 数値変換できない場合に返す
		 * @returns 変換した数値、できなければsub
		 */
		this.toNum = function(str, sub) {
			if(isNaN(str)) {
				return sub;
			}

			return Number(str);
		};
	};

	/**
	 * 午前午後の情報を保持するクラス。
	 * @class 午前午後クラス
	 */
	var Ampm = function(startHour, kanji, ascii) {
		this.StartHour = startHour;
		this.Kanji = kanji;
		this.Ascii = ascii;
	};

	/**
	 * 曜日の情報を保持するクラス。
	 * @class 曜日クラス
	 */
	var DayofWeek = function(day, kanji, ascii) {
		this.Day = day;
		this.Kanji = kanji;
		this.Ascii = ascii;
	};

	/**
	 * 元号の情報を保持するクラス。
	 * @class 元号クラス
	 */
	var Gengo = function(startDate, kanji, ascii, warekiMinus) {
		this.StartDate = startDate;
		this.Kanji = kanji;
		this.Ascii = ascii;
		this.WarekiMinus = warekiMinus;
	};

	/**
	 * 日付の解析情報を保持するクラス。
	 * @class 日付の解析情報クラス
	 */
	var ParsedDate = function() {
		this.YearType = false;		//年の書式指定文字に準ずる
		this.Gengo = false;
		this.Year = false;
		this.Month = 0;
		this.Day = 1;
		this.Hour = 0;
		this.Minute = 0;
		this.Second = 0;
		this.MilliSecond = 0;
		this.Ampm = false;

		this.Error = false;
		this.Parsed = false;
	};

	/**
	 *
	 * @class
	 * 日付の解析、変換等のユーティリティメソッドを提供する。<br />
	 * （メソッドはいずれもstaticで、本クラスをnewする必要はありません）
	 */
	this.DateUtil = new function() {

		//------privateメンバ--------------------------------------------------------------------------------------------------------

		/**
		 * 午前午後データ。
		 * @constant
		 */
		var _ampms = [
		                   new Ampm(12, "午後", "PM"),
			               new Ampm(0, "午前", "AM")
		              ];

		/**
		 * 曜日データ。
		 * @constant
		 */
		var _dayofWeeks = [
		                   new DayofWeek(0, "日", "Sunday"),
		                   new DayofWeek(1, "月", "Monday"),
		                   new DayofWeek(2, "火", "Tuesday"),
		                   new DayofWeek(3, "水", "Wednesday"),
		                   new DayofWeek(4, "木", "Thursday"),
		                   new DayofWeek(5, "金", "Friday"),
			               new DayofWeek(6, "土", "Saturday")
		              ];

		/**
		 * 元号データ。
		 * @constant
		 *
		 * 参考　元号の開始日
		 * 平成：1989/01/08
		 * 昭和：1926/12/25
		 * 大正：1912/07/30
		 * 明治：1868/10/23
		 */
		var _gengos = [
			               new Gengo(new Date(1989,  0,  8), "平成", "H", 1988),
			               new Gengo(new Date(1926, 11, 25), "昭和", "S", 1925),
			               new Gengo(new Date(1912,  6, 30), "大正", "T", 1911),
			               new Gengo(new Date(1868,  9, 23), "明治", "M", 1867)
		               ];

		/**
		 * 引数の時間が属する午前・午後を返す。
		 * @private
		 */
		var _getAmpm = function(hour) {
			var ret = null;
			var ampm = null;

			for (var i = 0; i < _ampms.length; i++) {
				ampm = _ampms[i];

			    if(ampm.StartHour <= hour) {
			    	ret = ampm;
			    	break;
			    }
			}

			if(ret == null) {
				ret = new Ampm(0, "不明", "?");
			}

			return ret;
		};

		/**
		 * 引数の時間が属する曜日を返す。
		 * @private
		 */
		var _getDayofWeek = function(day) {
			var ret = null;
			var dayofWeek = null;

			for (var i = 0; i < _dayofWeeks.length; i++) {
				dayofWeek = _dayofWeeks[i];

			    if(dayofWeek.Day == day) {
			    	ret = dayofWeek;
			    	break;
			    }
			}

			if(ret == null) {
				ret = new DayofWeek(0, "不明", "?");
			}

			return ret;
		};

		/**
		 * 引数の日付が属する元号を返す。
		 * @private
		 */
		var _getGengo = function(date) {
			var ret = null;
			var gengo = null;

			for (var i = 0; i < _gengos.length; i++) {
				gengo = _gengos[i];

			    if(gengo.StartDate <= date) {
			    	ret = gengo;
			    	break;
			    }
			}

			if(ret == null) {
				ret = new Gengo(new Date(), "不明", "?", 0);
			}

			return ret;
		};

		/**
		 * strの先頭が元号の漢字表現と一致すればそのGengoオブジェクトを返す。
		 * @private
		 */
		var _getGengoFromKanji = function(str, pos) {
			var ret = false;
			var gengo = null;

			for (var i = 0; i < _gengos.length; i++) {
				gengo = _gengos[i];

				if(StringUtil.startWith(str.substring(pos), gengo.Kanji)) {
					ret = gengo;
					break;
				}
			}

			return ret;
		};

		/**
		 * strの先頭が元号の漢字1文字表現と一致すればそのGengoオブジェクトを返す。
		 * @private
		 */
		var _getGengoFromKanji1 = function(str, pos) {
			var ret = false;
			var gengo = null;

			for (var i = 0; i < _gengos.length; i++) {
				gengo = _gengos[i];

				if(StringUtil.startWith(str.substring(pos), gengo.Kanji.substring(0, 1))) {
					ret = gengo;
					break;
				}
			}

			return ret;
		};

		/**
		 * strの先頭が元号のAscii表現と一致すればそのGengoオブジェクトを返す。
		 * @private
		 */
		var _getGengoFromAscii = function(str, pos) {
			var ret = false;
			var gengo = null;

			for (var i = 0; i < _gengos.length; i++) {
				gengo = _gengos[i];

				if(StringUtil.startWith(str.substring(pos), gengo.Ascii)) {
					ret = gengo;
					break;
				}
			}

			return ret;
		};

		/**
		 * strの先頭が午前・午後表現と一致すればそのAmpmオブジェクトを返す。
		 * @private
		 */
		var _getAmpmFromKanji = function(str, pos) {
			var ret = false;
			var ampm = null;

			for (var i = 0; i < _ampms.length; i++) {
				ampm = _ampms[i];

				if(StringUtil.startWith(str.substring(pos), ampm.Kanji)) {
					ret = ampm;
					break;
				}
			}

			return ret;
		};

		/**
		 * strの先頭がAM・PM表現と一致すればそのAmpmオブジェクトを返す。
		 * @private
		 */
		var _getAmpmFromAscii = function(str, pos) {
			var ret = false;
			var ampm = null;

			for (var i = 0; i < _ampms.length; i++) {
				ampm = _ampms[i];

				if(StringUtil.startWith(str.substring(pos), ampm.Ascii)) {
					ret = ampm;
					break;
				}
			}

			return ret;
		};

		/**
		 * strの先頭が曜日表現と一致すればそのDayofWeekオブジェクトを返す。
		 * @private
		 */
		var _getDayofWeekFromKanji = function(str, pos) {
			var ret = false;
			var dayofWeek = null;

			for (var i = 0; i < _dayofWeeks.length; i++) {
				dayofWeek = _dayofWeeks[i];

				if(StringUtil.startWith(str.substring(pos), dayofWeek.Kanji)) {
					ret = dayofWeek;
					break;
				}
			}

			return ret;
		};

		/**
		 * strの先頭が曜日表現と一致すればそのDayofWeekオブジェクトを返す。
		 * @private
		 */
		var _getDayofWeekFromAscii = function(str, pos) {
			var ret = false;
			var dayofWeek = null;

			for (var i = 0; i < _dayofWeeks.length; i++) {
				dayofWeek = _dayofWeeks[i];

				if(StringUtil.startWith(str.substring(pos), dayofWeek.Ascii)) {
					ret = dayofWeek;
					break;
				}
			}

			return ret;
		};

		/**
		 * strの先頭が数値ならその数値を返す。
		 * ketaは上限桁数。
		 * @private
		 */
		var _checkStartWithNumber = function(str, pos, keta) {
			var ret = "";
			var len = str.length;
			var moji;

			for (var i = pos; i < len; i++) {
				moji = str.charAt(i);

				if(isNaN(moji)) {
					break;
				} else {
					ret += moji;
				}

				if(i == pos + keta - 1) {
					break;
				}
			}

			if(ret == "") {
				ret = false;
			} else 	if(ret) {
				if(isNaN(ret)) {
					ret = false;
				}
			}

			return ret;
		};

		//------publicメンバ--------------------------------------------------------------------------------------------------------

		/**
		 * 加算タイプ(年)。
		 * @constant
		 */
		this.TYPE_YEAR = 0;

		/**
		 * 加算タイプ(月)。
		 * @constant
		 */
		this.TYPE_MONTH = 1;

		/**
		 * 加算タイプ(日)。
		 * @constant
		 */
		this.TYPE_DAY = 2;

		/**
		 * 加算タイプ(時間)。
		 * @constant
		 */
		this.TYPE_HOUR = 3;

		/**
		 * 加算タイプ(分)。
		 * @constant
		 */
		this.TYPE_MINUTE = 4;

		/**
		 * 加算タイプ(秒)。
		 * @constant
		 */
		this.TYPE_SECOND = 5;

		/**
		 * 日付を書式指定された文字列に変換する。
		 * @function
		 * @param date Date値
		 * @param format 書式指定文字列
		 * @returns 書式指定後の日付文字列
		 *
		 * @example
		 *  G4 : 元号（漢字2桁）
		 *  G2 : 元号（漢字1桁）
		 *  G1 : 元号（アルファベット1桁）
		 *  E2 : 年（和暦、2桁、0埋めあり）
		 *  E1 : 年（和暦、0埋めなし）
		 *  Y4 : 年（西暦、4桁）
		 *  Y2 : 年（西暦、2桁）
		 *  M2 : 月（2桁、0埋めあり）
		 *  M1 : 月（0埋めなし）
		 *  D2 : 日（2桁、0埋めあり）
		 *  D1 : 日（0埋めなし）
		 *  H2 : 時（24時間表記、2桁、0埋めあり）
		 *  H1 : 時（24時間表記、0埋めなし）
		 *  h2 : 時（12時間表記、2桁、0埋めあり）
		 *  h1 : 時（12時間表記、0埋めなし）
		 *  N2 : 分（2桁、0埋めあり）
		 *  N1 : 分（0埋めなし）
		 *  S2 : 秒（2桁、0埋めあり）
		 *  S1 : 秒（0埋めなし）
		 *  s3 : ミリ秒（3桁、0埋めあり）
		 *  s1 : ミリ秒（0埋めなし）
		 *  A4 : 午前 or午後
		 *  A2 : AM or PM
		 *  W4 : 曜日（日、月、火、水、木、金、土）
		 *  W2 : 曜日（Sunday,Monday,Tuesday,Wednesday,Thursday,Friday,Saturday）
		 *  #  : エスケープ文字。書式指定文字自体を指定したい場合に。
		 */
		this.format = function(date, format) {
			//空はそのまま返す
			if(CommonUtil.isEmpty(date)) {
				return date;
			}

			//日付でない場合はそのまま返す
			if(!this.isDate(date)) {
				return date;
			}

			var len = format.length;
			var syosiki = "";
			var ret = "";
			var clearFlag = false;

			for (var i = 0; i < len; i++) {
				syosiki += format.charAt(i);

				//書式指定文字が1文字では意味をなさないので基本的には次へスキップ
				if(syosiki.length == 1) {
					//最後の1文字が余った場合は単なる文字として付加
					if(i == len - 1) {
						ret += syosiki;
					}

					continue;
				}

				clearFlag = false;

				switch (syosiki.charAt(0)) {
					case "G":	//元号
						var gengo = _getGengo(date);

						clearFlag = true;
						switch (syosiki.charAt(1)) {
							case "4":	//漢字2桁
								ret += gengo.Kanji;
								break;
							case "2":	//漢字1桁
								ret += gengo.Kanji.charAt(0);
								break;
							case "1":	//アルファベット1桁
								ret += gengo.Ascii;
								break;
							default:	//書式指定文字ではなかった。そのまま文字として挿入。
								ret += syosiki.charAt(0);
								clearFlag = false;
								break;
						}

						break;
					case "E": 	//年（和暦）
						var gengo = _getGengo(date);
						var wareki = date.getFullYear() - gengo.WarekiMinus;

						clearFlag = true;
						switch (syosiki.charAt(1)) {
							case "2":	//2桁、0埋めあり
								ret += StringUtil.padLeft(wareki, "0", 2);
								break;
							case "1":	//0埋めなし
								ret += wareki;
								break;
							default:	//書式指定文字ではなかった。そのまま文字として挿入。
								ret += syosiki.charAt(0);
								clearFlag = false;
								break;
						}

						break;
					case "Y": 	//年（西暦）
						clearFlag = true;
						switch (syosiki.charAt(1)) {
							case "4":	//4桁
								ret += date.getFullYear();
								break;
							case "2":	//2桁
								ret += StringUtil.right(date.getFullYear(), 2);
								break;
							default:	//書式指定文字ではなかった。そのまま文字として挿入。
								ret += syosiki.charAt(0);
								clearFlag = false;
								break;
						}

						break;
					case "M": 	//月
						var month = date.getMonth() + 1;

						clearFlag = true;
						switch (syosiki.charAt(1)) {
							case "2":	//2桁、0埋めあり
								ret += StringUtil.padLeft(month, "0", 2);
								break;
							case "1":	//0埋めなし
								ret += month;
								break;
							default:	//書式指定文字ではなかった。そのまま文字として挿入。
								ret += syosiki.charAt(0);
								clearFlag = false;
								break;
						}

						break;
					case "D": 	//日
						var day = date.getDate();

						clearFlag = true;
						switch (syosiki.charAt(1)) {
							case "2":	//2桁、0埋めあり
								ret += StringUtil.padLeft(day, "0", 2);
								break;
							case "1":	//0埋めなし
								ret += day;
								break;
							default:	//書式指定文字ではなかった。そのまま文字として挿入。
								ret += syosiki.charAt(0);
								clearFlag = false;
								break;
						}

						break;
					case "H": 	//時（24時間表記）
						var hour = date.getHours();

						clearFlag = true;
						switch (syosiki.charAt(1)) {
							case "2":	//24時間表記、2桁、0埋めあり
								ret += StringUtil.padLeft(hour, "0", 2);
								break;
							case "1":	//24時間表記、0埋めなし
								ret += hour;
								break;
							default:	//書式指定文字ではなかった。そのまま文字として挿入。
								ret += syosiki.charAt(0);
								clearFlag = false;
								break;
						}

						break;
					case "h": 	//時（12時間表記）
						var hour = date.getHours();

						if(hour > 12) {
							hour -= 12;
						}

						clearFlag = true;
						switch (syosiki.charAt(1)) {
							case "2":	//24時間表記、2桁、0埋めあり
								ret += StringUtil.padLeft(hour, "0", 2);
								break;
							case "1":	//24時間表記、0埋めなし
								ret += hour;
								break;
							default:	//書式指定文字ではなかった。そのまま文字として挿入。
								ret += syosiki.charAt(0);
								clearFlag = false;
								break;
						}

						break;
					case "N": 	//分
						var minute = date.getMinutes();

						clearFlag = true;
						switch (syosiki.charAt(1)) {
							case "2":	//2桁、0埋めあり
								ret += StringUtil.padLeft(minute, "0", 2);
								break;
							case "1":	//0埋めなし
								ret += minute;
								break;
							default:	//書式指定文字ではなかった。そのまま文字として挿入。
								ret += syosiki.charAt(0);
								clearFlag = false;
								break;
						}

						break;
					case "S": 	//秒
						var second = date.getSeconds();

						clearFlag = true;
						switch (syosiki.charAt(1)) {
							case "2":	//2桁、0埋めあり
								ret += StringUtil.padLeft(second, "0", 2);
								break;
							case "1":	//0埋めなし
								ret += second;
								break;
							default:	//書式指定文字ではなかった。そのまま文字として挿入。
								ret += syosiki.charAt(0);
								clearFlag = false;
								break;
						}

						break;
					case "s": 	//ミリ秒
						var milliSecond = date.getMilliseconds();

						clearFlag = true;
						switch (syosiki.charAt(1)) {
							case "3":	//3桁、0埋めあり
								ret += StringUtil.padLeft(milliSecond, "0", 3);
								break;
							case "1":	//0埋めなし
								ret += milliSecond;
								break;
							default:	//書式指定文字ではなかった。そのまま文字として挿入。
								ret += syosiki.charAt(0);
								clearFlag = false;
								break;
						}

						break;
					case "A": 	//午前 or午後
						var hour = date.getHours();
						var ampm = _getAmpm(hour);

						clearFlag = true;
						switch (syosiki.charAt(1)) {
							case "4":	//午前 or午後
								ret += ampm.Kanji;
								break;
							case "2":	//AM or PM
								ret += ampm.Ascii;
								break;
							default:	//書式指定文字ではなかった。そのまま文字として挿入。
								ret += syosiki.charAt(0);
								clearFlag = false;
								break;
						}

						break;
					case "W": 	//曜日
						var day = date.getDay();
						var dayofWeek = _getDayofWeek(day);

						clearFlag = true;
						switch (syosiki.charAt(1)) {
							case "4":	//漢字
								ret += dayofWeek.Kanji;
								break;
							case "2":	//英語
								ret += dayofWeek.Ascii;
								break;
							default:	//書式指定文字ではなかった。そのまま文字として挿入。
								ret += syosiki.charAt(0);
								clearFlag = false;
								break;
						}

						break;
					case "#": 	//固定文字
						ret += syosiki.charAt(1);
						clearFlag = true;
						break;
					default:
						ret += syosiki.charAt(0);
						break;
				}

				//2文字単位で処理できた場合はすべてクリア
				//できなかった場合は1文字だけ処理したと見なし1文字ずらす
				if(clearFlag) {
					syosiki = "";
				} else {
					syosiki = syosiki.charAt(1);
				}

				//最後の1文字が余った場合は単なる文字として付加
				if(syosiki.length == 1 && i == len - 1) {
					ret += syosiki;
				}
			}

			return ret;
		};


		/**
		 * 文字列を指定された書式に基づいて日付に変換する。
		 * 書式指定文字列は{@link SysketLibrary.DateUtil.format}参照。
		 * @function
		 * @param str 変換対象文字列
		 * @param format 書式指定文字列({@link SysketLibrary.DateUtil.format}参照)
		 * @param check 厳密な形式チェックをするか？
		 * 				 trueの場合は書式指定文字以外の文字が後続にある場合にパースエラーとする。
		 * @returns 変換後の日付
		 */
		this.parse = function(str, format, check) {
			//空はそのまま返す
			if(CommonUtil.isEmpty(str)) {
				return str;
			}

			var len = format.length;
			var syosiki = "";
			var clearFlag = false;
			var pos = 0;
			var p = new ParsedDate();

			if(check == undefined) {
				check = false;
			}

			for (var i = 0; i < len; i++) {
				//書式指定文字の空白はスキップ
				if(format.charAt(i) == " ") {
					continue;
				}

				syosiki += format.charAt(i);

				//書式指定文字が1文字では意味をなさないので基本的には次へスキップ
				if(syosiki.length == 1) {
					continue;
				}

				//解析対象も空白はスキップ
				pos = StringUtil.skipSpace(str, pos);
				clearFlag = false;

				switch (syosiki.charAt(0)) {
					case "G":	//元号
						var gengo;

						//すでに年が他のタイプと解析されていればエラー
						if(p.YearType) {
							p.Error = true;
						}

						clearFlag = true;

						switch (syosiki.charAt(1)) {
							case "4":	//漢字2桁
								gengo = _getGengoFromKanji(str, pos);
								if(gengo) {
									p.Gengo = gengo;
									pos += gengo.Kanji.length;
								} else {
									p.Error = true;
								}

								break;
							case "2":	//漢字1桁
								gengo = _getGengoFromKanji1(str, pos);
								if(gengo) {
									p.Gengo = gengo;
									pos++;
								} else {
									p.Error = true;
								}

								break;
							case "1":	//アルファベット1桁
								gengo = _getGengoFromAscii(str, pos);
								if(gengo) {
									p.Gengo = gengo;
									pos += gengo.Ascii.length;
								} else {
									p.Error = true;
								}

								break;
							default:	//書式指定文字ではなかった。
								clearFlag = false;
								break;
						}

						if(!p.Parsed) {p.Parsed = clearFlag;}
						break;

					case "E": 	//年（和暦）
						clearFlag = true;
						p.YearType = syosiki.charAt(0);

						switch (syosiki.charAt(1)) {
							case "2":	//2桁、0埋めあり
								p.Year = _checkStartWithNumber(str, pos, 2);
								pos += ("" + p.Year).length;
								break;
							case "1":	//0埋めなし
								p.Year = _checkStartWithNumber(str, pos, 2);
								pos += ("" + p.Year).length;
								break;
							default:	//書式指定文字ではなかった。
								clearFlag = false;
								break;
						}

						//指定桁数の数値が取得できなかった場合はフォーマットエラー
						if(clearFlag && !p.Year) {
							p.Error = true;
						}

						if(!p.Parsed) {p.Parsed = clearFlag;}
						break;

					case "Y": 	//年（西暦）
						clearFlag = true;
						p.YearType = syosiki.charAt(0);

						switch (syosiki.charAt(1)) {
							case "4":	//4桁
								p.Year = _checkStartWithNumber(str, pos, 4);
								pos += ("" + p.Year).length;
								break;
							case "2":	//2桁
								p.Year = _checkStartWithNumber(str, pos, 2);
								pos += ("" + p.Year).length;
								break;
							default:	//書式指定文字ではなかった。
								clearFlag = false;
								break;
						}

						//指定桁数の数値が取得できなかった場合はフォーマットエラー
						if(clearFlag && !p.Year) {
							p.Error = true;
						}

						if(!p.Parsed) {p.Parsed = clearFlag;}
						break;

					case "M": 	//月
						clearFlag = true;
						switch (syosiki.charAt(1)) {
							case "2":	//2桁、0埋めあり
							case "1":	//0埋めなし
								p.Month = _checkStartWithNumber(str, pos, 2);
								pos += ("" + p.Month).length;
								break;
							default:	//書式指定文字ではなかった。そのまま文字として挿入。
								clearFlag = false;
								break;
						}

						//指定桁数の数値が取得できなかった場合はフォーマットエラー
						if(clearFlag && !p.Month) {
							p.Error = true;
						}

						if(!p.Parsed) {p.Parsed = clearFlag;}
						break;

					case "D": 	//日
						clearFlag = true;
						switch (syosiki.charAt(1)) {
							case "2":	//2桁、0埋めあり
							case "1":	//0埋めなし
								p.Day = _checkStartWithNumber(str, pos, 2);
								pos += ("" + p.Day).length;
								break;
							default:	//書式指定文字ではなかった。そのまま文字として挿入。
								clearFlag = false;
								break;
						}

						//指定桁数の数値が取得できなかった場合はフォーマットエラー
						if(clearFlag && !p.Day) {
							p.Error = true;
						}

						if(!p.Parsed) {p.Parsed = clearFlag;}
						break;

					case "H": 	//時（24時間表記）
					case "h": 	//時（12時間表記）
						clearFlag = true;
						switch (syosiki.charAt(1)) {
							case "2":	//24時間表記、2桁、0埋めあり
							case "1":	//24時間表記、0埋めなし
								p.Hour = _checkStartWithNumber(str, pos, 2);
								pos += ("" + p.Hour).length;
								break;
							default:	//書式指定文字ではなかった。そのまま文字として挿入。
								clearFlag = false;
								break;
						}

						if(!p.Parsed) {p.Parsed = clearFlag;}
						break;

					case "N": 	//分
						clearFlag = true;
						switch (syosiki.charAt(1)) {
							case "2":	//2桁、0埋めあり
							case "1":	//0埋めなし
								p.Minute = _checkStartWithNumber(str, pos, 2);
								pos += ("" + p.Minute).length;
								break;
							default:	//書式指定文字ではなかった。そのまま文字として挿入。
								clearFlag = false;
								break;
						}

						if(!p.Parsed) {p.Parsed = clearFlag;}
						break;

					case "S": 	//秒
						clearFlag = true;
						switch (syosiki.charAt(1)) {
							case "2":	//2桁、0埋めあり
							case "1":	//0埋めなし
								p.Second = _checkStartWithNumber(str, pos, 2);
								pos += ("" + p.Second).length;
								break;
							default:	//書式指定文字ではなかった。そのまま文字として挿入。
								clearFlag = false;
								break;
						}

						if(!p.Parsed) {p.Parsed = clearFlag;}
						break;

					case "s": 	//ミリ秒
						clearFlag = true;
						switch (syosiki.charAt(1)) {
							case "3":	//3桁、0埋めあり
							case "1":	//0埋めなし
								p.MilliSecond = _checkStartWithNumber(str, pos, 3);
								pos += ("" + p.MilliSecond).length;
								break;
							default:	//書式指定文字ではなかった。そのまま文字として挿入。
								clearFlag = false;
								break;
						}

						if(!p.Parsed) {p.Parsed = clearFlag;}
						break;

					case "A": 	//午前 or午後
						var ampm;

						//すでにAM・PMが解析されていればエラー
						if(p.Ampm) {
							p.Error = true;
						}

						clearFlag = true;
						switch (syosiki.charAt(1)) {
							case "4":	//午前 or午後
								ampm = _getAmpmFromKanji(str, pos);
								if(ampm) {
									p.Ampm = ampm;
									pos += ampm.Kanji.length;
								} else {
									p.Error = true;
								}

								break;
							case "2":	//AM or PM
								ampm = _getAmpmFromAscii(str, pos);
								if(ampm) {
									p.Ampm = ampm;
									pos += ampm.Ascii.length;
								} else {
									p.Error = true;
								}

								break;
							default:	//書式指定文字ではなかった。そのまま文字として挿入。
								clearFlag = false;
								break;
						}

						break;

					case "W": 	//曜日
						var dayofWeek;

						clearFlag = true;
						switch (syosiki.charAt(1)) {
							case "4":	//漢字
								dayofWeek = _getDayofWeekFromKanji(str, pos);
								if(dayofWeek) {
									pos += dayofWeek.Kanji.length;
								} else {
									p.Error = true;
								}

								break;
							case "2":	//英語
								dayofWeek = _getDayofWeekFromAscii(str, pos);
								if(dayofWeek) {
									pos += dayofWeek.Ascii.length;
								} else {
									p.Error = true;
								}

								break;
							default:	//書式指定文字ではなかった。
								clearFlag = false;
								break;
						}

						break;
					case "#": 	//固定文字
						if(StringUtil.startWith(str.substring(pos), syosiki.charAt(1))) {
							pos++;
							clearFlag = true;
						} else {
							p.Error = true;
						}

						break;

					default:
						//文字列の該当位置が書式指定文字そのものか？
						if(StringUtil.startWith(str.substring(pos), syosiki.charAt(0))) {
							//そのものであれば1文字だけ処理したと見なす
							pos++;
						} else {
							//そのものではない。フォーマットエラー。
							p.Error = true;
						}

						break;
				}

				//alert(syosiki + " : " + pos + " : " + p.Error);

				//2文字単位で処理できた場合はすべてクリア
				if(clearFlag) {
					syosiki = "";
				} else {
					syosiki = syosiki.charAt(1);
				}

				//解析中にエラーがあればfalseを返す
				if(p.Error) {
					return false;
				}
			}

			//書式指定文字列以降にまだ余計な文字がある場合はエラー扱い
			if(check && str.length > pos) {
				return false;
			}

			//年月日時分秒のいずれも解析されていない場合はエラー扱い
			if(!p.Parsed) {
				return false;
			}

			//解析結果を数値に変換
			p.Year = NumberUtil.toNum(p.Year);
			p.Month = NumberUtil.toNum(p.Month);
			p.Day = NumberUtil.toNum(p.Day);
			p.Hour = NumberUtil.toNum(p.Hour);
			p.Minute = NumberUtil.toNum(p.Minute);
			p.Second = NumberUtil.toNum(p.Second);
			p.MilliSecond = NumberUtil.toNum(p.MilliSecond);

			if(p.Month) {
				p.Month--;
			}

			//解析情報から日付を再構築
			var year = p.Year;

			switch (p.YearType) {
				case "E":
					if(p.Gengo) {
						year += p.Gengo.WarekiMinus;
					} else {
						//Eが指定されたのにGがない場合は元号が分からないのでフォーマットエラー
						p.Error = true;
					}

					break;
				case "Y":
					if(year < 100) {
						if(year < 30) {
							year += 2000;
						} else {
							year += 1900;
						}
					}

					break;
				default:
					//年が指定されていない場合は現在年を使用する
					year = new Date().getFullYear();
					break;
			}

			//解析情報から午前・午後を反映
			if(p.Ampm) {
				if(p.Hour < p.Ampm.StartHour) {
					p.Hour += p.Ampm.StartHour;
				}
			}

			//解析中にエラーがあればfalseを返す
			if(p.Error) {
				return false;
			}

			//alert(year + ":" + p.Month + ":" + p.Day + ":" + p.Hour + ":" + p.Minute + ":" + p.Second + ":" + p.MilliSecond);

			//javascriptのdate関数で成型された日付と、入力された日付の比較

			var date = new Date(year, p.Month, p.Day, p.Hour, p.Minute, p.Second, p.MilliSecond);

			//alert("入力値 :" + year + "/" + p.Month + "/" + p.Day + "/" + p.Hour + "/" + p.Minute + "/" + p.Second + "/" + p.MilliSecond);
			//alert("date関数 :" + date.getFullYear() + "/" + date.getMonth() + "/" + date.getDate() + "/" + date.getHours() + "/" + date.getMinutes() + "/" + date.getSeconds() + "/" + date.getMilliseconds());

			if(date.getFullYear() != year){
				return false;
			}

			if( date.getMonth() != p.Month){
				return false;
			}

			if(date.getDate() != p.Day){
				return false;
			}

			if(date.getHours() != p.Hour){
				return false;
			}

			if(date.getMinutes() != p.Minute){
				return false;
			}

			if(date.getSeconds() != p.Second ){
				return false;
			}

			if(date.getMilliseconds() != p.MilliSecond){
				return false;
			}


			return date;
		};

		/**
		 * 日付型かどうかの判定
		 * @function
		 * @param date 判定対象
		 * @returns true:日付型である、false:日付型でない
		 */
		this.isDate = function(date) {
			return date instanceof Date;
		};

		/**
		 * parseして日付と解釈できるかどうかの判定
		 * @function
		 * @param str 判定対象
		 * @param format 書式指定文字列({@link SysketLibrary.DateUtil.format}参照)
		 * @returns true:日付と解釈できる、false:日付と解釈できない
		 */
		this.isDateString = function(str, format) {
			if(CommonUtil.isEmpty(str)) {
				return true;
			}

			return this.isDate(this.parse(str, format, true));
		};

		/**
		 * 年を加算
		 * @function
		 * @param date 対象日付
		 * @param num 加算数
		 * @returns dateが日付型なら加算した日付、日付型でない場合は引数をそのまま返す
		 */
		this.addYear = function(date, num) {
			return this.addDate(date, this.TYPE_YEAR, num);
		};

		/**
		 * 月を加算
		 * @function
		 * @param date 対象日付
		 * @param num 加算数
		 * @returns dateが日付型なら加算した日付、日付型でない場合は引数をそのまま返す
		 */
		this.addMonth = function(date, num) {
			return this.addDate(date, this.TYPE_MONTH, num);
		};

		/**
		 * 日を加算
		 * @function
		 * @param date 対象日付
		 * @param num 加算数
		 * @returns dateが日付型なら加算した日付、日付型でない場合は引数をそのまま返す
		 */
		this.addDay = function(date, num) {
			return this.addDate(date, this.TYPE_DAY, num);
		};

		/**
		 * 時間を加算
		 * @function
		 * @param date 対象日付
		 * @param num 加算数
		 * @returns dateが日付型なら加算した日付、日付型でない場合は引数をそのまま返す
		 */
		this.addHour = function(date, num) {
			return this.addDate(date, this.TYPE_HOUR, num);
		};

		/**
		 * 分を加算
		 * @function
		 * @param date 対象日付
		 * @param num 加算数
		 * @returns dateが日付型なら加算した日付、日付型でない場合は引数をそのまま返す
		 */
		this.addMinute = function(date, num) {
			return this.addDate(date, this.TYPE_MINUTE, num);
		};

		/**
		 * 秒を加算
		 * @function
		 * @param date 対象日付
		 * @param num 加算数
		 * @returns dateが日付型なら加算した日付、日付型でない場合は引数をそのまま返す
		 */
		this.addSecond = function(date, num) {
			return this.addDate(date, this.TYPE_SECOND, num);
		};

		/**
		 * 指定したタイプの時間を加算
		 * @function
		 * @param date 対象日付
		 * @param type 加算する時間のタイプ。クラス定義のTYPE～定数で指定。
		 * @param num 加算数
		 * @returns dateが日付型、かつtypeが有効なら加算した日付、それ以外の場合は引数をそのまま返す
		 */
		this.addDate = function(date, type, num) {
			//日付でない場合はそのまま返す
			if(!this.isDate(date)) {
				return date;
			}

			var d = new Date();
			var add = num;

			if(type == this.TYPE_YEAR) {
				//年の場合

				d.setTime(date.getTime());
				d.setFullYear(date.getFullYear() + num);
			} else if(type == this.TYPE_MONTH) {
				//月の場合

				var month = date.getMonth();
				var addYear = 0;

				month = month + num;

				while(month > 11 || month < 0) {
					if(month > 11) {
						month -= 12;
						addYear++;
					} else if(month < 0) {
						month += 12;
						addYear--;
					}
				}

				d.setTime(date.getTime());
				d.setMonth(month);

				if(addYear != 0) {
					d.setFullYear(d.getFullYear() + addYear);
				}

				//月末日考慮(例えば1/31の1ヶ月後が2/31ではなく3/3になるような場合)
				if(d.getMonth() > month) {
					d = this.getMonthFirst(d);
					d = this.addDay(d, -1);
				}
			} else {
				//年、月以外はミリ秒に換算できる

				if(type <= this.TYPE_SECOND) {
					add = add * 1000;
				}

				if(type <= this.TYPE_MINUTE) {
					add = add * 60;
				}

				if(type <= this.TYPE_HOUR) {
					add = add * 60;
				}

				if(type <= this.TYPE_DAY) {
					add = add * 24;
				}

				d.setTime(date.getTime() + add);
			}

			return d;
		};

		/**
		 * 年初日を返す
		 * @function
		 * @param date 対象日付
		 * @returns dateが日付型なら加算した年初日、日付型でない場合は引数をそのまま返す
		 */
		this.getYearFirst = function(date) {
			//日付でない場合はそのまま返す
			if(!this.isDate(date)) {
				return date;
			}

			var d = new Date();
			d.setTime(date.getTime());
			d.setMonth(0);
			d.setDate(1);

			return d;
		};

		/**
		 * 年末日を返す
		 * @function
		 * @param date 対象日付
		 * @returns dateが日付型なら加算した年末日、日付型でない場合は引数をそのまま返す
		 */
		this.getYearLast = function(date) {
			//日付でない場合はそのまま返す
			if(!this.isDate(date)) {
				return date;
			}

			var d = new Date();
			d.setTime(date.getTime());
			d.setFullYear(d.getFullYear() + 1);
			d.setMonth(0);
			d.setDate(1);
			d = this.addDay(d, -1);

			return d;
		};

		/**
		 * 月初日を返す
		 * @function
		 * @param date 対象日付
		 * @returns dateが日付型なら加算した月初日、日付型でない場合は引数をそのまま返す
		 */
		this.getMonthFirst = function(date) {
			//日付でない場合はそのまま返す
			if(!this.isDate(date)) {
				return date;
			}

			var d = new Date();
			d.setTime(date.getTime());
			d.setDate(1);

			return d;
		};

		/**
		 * 月末日を返す
		 * @function
		 * @param date 対象日付
		 * @returns dateが日付型なら加算した月末日、日付型でない場合は引数をそのまま返す
		 */
		this.getMonthLast = function(date) {
			//日付でない場合はそのまま返す
			if(!this.isDate(date)) {
				return date;
			}

			var d = new Date();
			d.setTime(date.getTime());
			d = this.addMonth(d, 1);
			d = this.getMonthFirst(d);
			d = this.addDay(d, -1);

			return d;
		};
	};

	SysketLibrary = this;

})();

